package gov.va.vamf.scheduling.direct.resource;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.veteranappointment.resources.AbstractUserResource;
import gov.va.vamf.scheduling.direct.datalayer.notification.NotificationDataService;
import gov.va.vamf.scheduling.direct.domain.InAppMessage;
import gov.va.vamf.security.v1.filters.JwtResourceRestricted;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.ws.rs.*;
import java.util.List;

@Path("/notification-service")
@Component
@Scope("request")
@JwtResourceRestricted
public class NotificationResource extends AbstractUserResource {

    @Resource
    NotificationDataService notificationDataService;

    /**
     * Fetch in-app notifications for patient
     *
     * @param assigningAuthority
     * @param patientId
     * @return List of InAppMessage
     */
    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/notifications")
    @Produces({"application/xml", "application/json"})
    public List<InAppMessage> fetchNotifications(@PathParam("assigning-authority") String assigningAuthority,
                                                 @PathParam("patient-id") String patientId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return notificationDataService.fetchNotifications(patientIdentifier) ;
    }

    @PUT
    @Path("/patient/{assigning-authority}/{patient-id}/notifications/{notification-id}/read")
    public void setNotificationAsRead(@PathParam("assigning-authority") String assigningAuthority,
                                      @PathParam("patient-id") String patientId,
                                      @PathParam("notification-id") String notificationId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        if (patientOwnsNotification(patientIdentifier, notificationId)) {
            notificationDataService.setNotificationAsRead(notificationId);
        } else {
            final String errText = String.format("%s attempted to mark as read message at %s without owning it", patientIdentifier, notificationId);
        }
    }

    @DELETE
    @Path("/patient/{assigning-authority}/{patient-id}/notifications/{notification-id}")
    public void deleteNotification(@PathParam("assigning-authority") String assigningAuthority,
                                   @PathParam("patient-id") String patientId,
                                   @PathParam("notification-id") String notificationId){

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        if( patientOwnsNotification(patientIdentifier,notificationId) ) {
            notificationDataService.deleteNotification(notificationId);
        } else {
            final String errText = String.format("%s attempted to delete message at %s without owning it", patientIdentifier, notificationId);
            throw new ForbiddenException(errText);
        }
    }

    private boolean patientOwnsNotification(PatientIdentifier patientIdentifier, String notificationId){
        List<InAppMessage> vetMessages = notificationDataService.fetchNotifications(patientIdentifier);
        for(InAppMessage message : vetMessages) {
            if(message.get_id().equals(notificationId)){
                return true;
            }
        }
        return false;
    }
}
